/*
*  Arnold emulator (c) Copyright, Kevin Thacker 1995-2015
*
*  This file is part of the Arnold emulator source code distribution.
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include "cpc.h"
#include "emudevice.h"
#include "w29c040.h"
/* TotO's comments in cpcwiki:

ODD ($7Fxx) = X-MEM
EVEN ($7Exx) = CPC 6128 Expansion or Y-MEM
*/
/* NEEDS work! */
extern unsigned char *Z80MemoryBase;

typedef struct
{
	BOOL XMem;
	int RomSelect;
	int RamSelect;
	w29c040_flash Rom;
	unsigned char MemFlashRom[512 * 1024];
	unsigned char MemRam[512 * 1024];
	BOOL RamSwitchState;
	BOOL BootSwitchState;
	BOOL RomLockSwitchState;
	BOOL ReadRomSwitchState;
	ExpansionRomData m_Roms;
	DkRamData m_RamData;
} XYMem;

static XYMem XMem;
static XYMem YMem;

BOOL XYMem_GetRamSwitch(XYMem *pMem)
{
	return pMem->RamSwitchState;
}
void XYMem_SetRamSwitch(XYMem *pMem, BOOL bState)
{
	pMem->RamSwitchState = bState;
}


BOOL XYMem_GetBootSwitch(XYMem *pMem)
{
	return pMem->BootSwitchState;
}

void XYMem_SetBootSwitch(XYMem *pMem,BOOL bState)
{
	pMem->BootSwitchState = bState;
}


BOOL XYMem_GetRomProtectSwitch(XYMem *pMem)
{
	return pMem->RomLockSwitchState;
}

void XYMem_SetRomProtectSwitch(XYMem *pMem,BOOL bState)
{
	pMem->RomLockSwitchState = bState;
}



BOOL XYMem_GetRomEnabledSwitch(XYMem *pMem)
{
	return pMem->ReadRomSwitchState;
}

void XYMem_SetRomEnabledSwitch(XYMem *pMem,BOOL bState)
{
	pMem->ReadRomSwitchState = bState;
}

void XYMem_Reset(XYMem *pMem)
{
	pMem->RamSelect = 0;
	pMem->RomSelect = 0;
}

void	XYMem_ROM_Write(XYMem *pMem, Z80_WORD Port, Z80_BYTE Data)
{
	pMem->RomSelect = Data;
	
    Computer_RethinkMemory();	
}

void XYMem_Flash_Write(XYMem *pMem, Z80_WORD Addr, Z80_BYTE Data)
{
	int Flash16KBPage = ((Addr >> 14) & 0x03) | ((pMem->RomSelect & 0x03f) << 2);

	// todo: handle commands
	pMem->MemFlashRom[(Flash16KBPage << 14) | (Addr & 0x03fff)] = Data;
}

void	XYMem_RAM_Write(XYMem *pMem, Z80_WORD Port, Z80_BYTE Data)
{
	if ((Data & 0x0c0) == 0x0c0)
	{
		pMem->RamSelect = Data;

		Computer_RethinkMemory();
	}
}


void XYMem_MemoryRethink(XYMem *pMem, MemoryData *pData)
{
	int RamBank = ((pMem->RamSelect >> 3) & 0x07);

	switch (pMem->RamSelect & 0x07)
	{
		/* nothing */
	case 0:
		break;

	case 2:
	{
		int i;
		/* complete switch */
		for (i = 0; i < 4; i++)
		{
			int nPage = i;
			int nRamOffset = (i << 14);
			int nOffset = nRamOffset - (nPage << 14);
			pData->pWritePtr[(i << 1) + 0] = pMem->MemRam + (RamBank << 16) - nOffset;
			pData->pWritePtr[(i << 1) + 1] = pData->pWritePtr[(i << 1) + 0];
			if (!pData->bRomEnable[(i << 1) + 0])
			{
				pData->pReadPtr[(i << 1) + 0] = pData->pWritePtr[(i << 1) + 0];
				pData->pReadMaskPtr[(i << 1) + 0] = GetDefaultReadMask() - (i<<14);
			}

			if (!pData->bRomEnable[(i << 1) + 1])
			{
				pData->pReadPtr[(i << 1) + 1] = pData->pWritePtr[(i << 1) + 1];
				pData->pReadMaskPtr[(i << 1) + 1] = GetDefaultReadMask() - (i << 14);
			}
			pData->bRamDisable[(i << 1) + 0] = TRUE;
			pData->bRamDisable[(i << 1) + 1] = TRUE;
		}
	}
	break;

	case 1:
	{
		pData->pWritePtr[6] = (pMem->MemRam + (RamBank << 16)) + (3 << 14) - 0x0c000;
		pData->pWritePtr[7] = pData->pWritePtr[6];
		if (!pData->bRomEnable[7])
		{
			pData->pReadPtr[6] = pData->pWritePtr[6];
			pData->pReadMaskPtr[6] = GetDefaultReadMask() - 0x0c000;
		}
		if (!pData->bRomEnable[6])
		{
			pData->pReadPtr[7] = pData->pWritePtr[7];
			pData->pReadMaskPtr[7] = GetDefaultReadMask() - 0x0c000;
		}
		pData->bRamDisable[6] = TRUE;
		pData->bRamDisable[7] = TRUE;
	}
	break;

	case 3:
	{
		/* pal can do this because it swaps a14/a15 to ram */

		/* page 7 appears at position 3 */
		pData->pWritePtr[7] = pMem->MemRam + (RamBank << 16) + (3 << 14) - 0x0c000;
		pData->pWritePtr[6] = pData->pWritePtr[7];
		if (!pData->bRomEnable[7])
		{
			pData->pReadPtr[7] = pData->pWritePtr[7];
			pData->pReadMaskPtr[7] = GetDefaultReadMask() - 0x0c000;
		}
		if (!pData->bRomEnable[6])
		{
			pData->pReadPtr[6] = pData->pWritePtr[6];
			pData->pReadMaskPtr[6] = GetDefaultReadMask() - 0x0c000;
		}
		pData->bRamDisable[7] = TRUE;
		pData->bRamDisable[6] = TRUE;

		/* normal page 3 ram appears in range &4000-&7fff */
		pData->pWritePtr[2] = Z80MemoryBase + (3 << 14) - 0x04000;
		pData->pWritePtr[3] = pData->pWritePtr[2];

		pData->pReadPtr[2] = pData->pWritePtr[2];
		pData->pReadPtr[3] = pData->pWritePtr[3];
		pData->pReadMaskPtr[2] = GetDefaultReadMask() - 0x04000;
		pData->pReadMaskPtr[3] = GetDefaultReadMask() - 0x04000;
		pData->bRamDisable[2] = TRUE;
		pData->bRamDisable[3] = TRUE;
	}
	break;

	case 4:
	case 5:
	case 6:
	case 7:
	{
		/* 4000-7fff only */
		pData->pWritePtr[2] = pMem->MemRam + (RamBank << 16) + ((pMem->RamSelect & 0x03) << 14) - 0x04000;
		pData->pWritePtr[3] = pData->pWritePtr[2];

		pData->pReadPtr[2] = pData->pWritePtr[2];
		pData->pReadPtr[3] = pData->pWritePtr[3];
		pData->pReadMaskPtr[2] = GetDefaultReadMask() - 0x04000;
		pData->pReadMaskPtr[3] = GetDefaultReadMask() - 0x04000;
		pData->bRamDisable[2] = TRUE;
		pData->bRamDisable[3] = TRUE;
	}
	break;
	}


	/* boot from x-mem and rom enabled? */
	/* TODO: Check, is rom 0 also disabled? */
	if (pMem->ReadRomSwitchState && pMem->BootSwitchState)
	{
		/* firmware is visible */
		const unsigned char *pRomData = &pMem->MemFlashRom[(7 << 14)];
		
		pData->pReadPtr[0] = pRomData;
		pData->pReadPtr[1] = pRomData;
		pData->pReadMaskPtr[0] = GetDefaultReadMask() - 0x0;
		pData->pReadMaskPtr[1] = GetDefaultReadMask() - 0x0;
		pData->bRomDisable[1] = TRUE;
		pData->bRomDisable[0] = TRUE;
	}
	
	/* To access flash fully, you need to use &40 */
	/* TODO: Check that flash can be accessed when boot from CPC is set AND we have a rom that can read from tape */
	
    if (pMem->ReadRomSwitchState &&
		/* 010xxxxx - all ROMS are readable &40-&50 */
		(((pMem->RomSelect & 0x0e0)==0x040) ||
		/* 000xxxxx - all ROMS except 7 are readable &00-&1f */
		(((pMem->RomSelect & 0x0e0)==0x0) && ((pMem->RomSelect&0x01f)!=7)))
		)
		{
		const unsigned char *pRomData = &pMem->MemFlashRom[(pMem->RomSelect & 0x01f) << 14]-0x0c000;
		
		if (pData->bRomEnable[6] && !pData->bRomDisable[6])
		{
			pData->bRomDisable[6] = TRUE;
			pData->pReadPtr[6] = pRomData;
			pData->pReadMaskPtr[6] = GetDefaultReadMask() - 0x0c0000;
		}
		if (pData->bRomEnable[7] && !pData->bRomDisable[7])
		{
			pData->bRomDisable[7] = TRUE;
			pData->pReadPtr[7] = pRomData;
			pData->pReadMaskPtr[7] = GetDefaultReadMask() - 0x0c000;
		}
    }
}

void XYMem_ClearExpansionRom(XYMem *pMem, int RomIndex)
{
	memset(pMem->MemFlashRom + (RomIndex << 14), 0x0ff, 16384);
}

BOOL XYMem_SetExpansionRom(XYMem *pMem, int RomIndex, const unsigned char *pData, unsigned long Length)
{
	EmuDevice_CopyRomData(pMem->MemFlashRom + (RomIndex << 14), 16384, pData, Length);
	return TRUE;
}


// TODO
void XMem_ClearExpansionRom(int RomIndex)
{
	XYMem_ClearExpansionRom(&XMem, RomIndex);
}

// TODO
BOOL XMem_SetExpansionRom(int RomIndex, const unsigned char *pData, unsigned long Length)
{
	XYMem_SetExpansionRom(&XMem, RomIndex, pData, Length);
	return TRUE;
}


void XYMem_InitDevice(XYMem *pMem, BOOL IsXMem)
{
	int i;

	/* find actual initialised value. Any value other than 0 should be ok including ff*/
	memset(pMem->MemFlashRom, 0x01, sizeof(pMem->MemFlashRom));
	memset(pMem->MemRam, 0x0ff, sizeof(pMem->MemRam));

	// TODO
	ExpansionRom_Init(&pMem->m_Roms, XMem_ClearExpansionRom, XMem_SetExpansionRom);

	for (i = 0; i < 32; i++)
	{
		ExpansionRom_SetAvailableState(&pMem->m_Roms, i, i!=7 ? TRUE : FALSE);
	}
	for (i = 0; i < MAX_DKRAM_PAGES; i++)
	{
		pMem->m_RamData.PageAvailable[i] = TRUE;
		pMem->m_RamData.Pages[i] = pMem->MemRam + (i << 14);
	}	
}


BOOL XMem_GetRamSwitch(void)
{
	return XYMem_GetRamSwitch(&XMem);
}

void XMem_SetRamSwitch(BOOL bState)
{
	XYMem_SetRamSwitch(&XMem, bState);
}


BOOL XMem_GetBootSwitch(void)
{
	return XYMem_GetBootSwitch(&XMem);
}

void XMem_SetBootSwitch(BOOL bState)
{
	XYMem_SetBootSwitch(&XMem, bState);
}


BOOL XMem_GetRomProtectSwitch(void)
{
	return XYMem_GetRomProtectSwitch(&XMem);
}

void XMem_SetRomProtectSwitch(BOOL bState)
{
	XYMem_SetRomProtectSwitch(&XMem, bState);
}



BOOL XMem_GetRomEnabledSwitch(void)
{
	return XYMem_GetRomEnabledSwitch(&XMem);
}

void XMem_SetRomEnabledSwitch(BOOL bState)
{
	XYMem_SetRomEnabledSwitch(&XMem, bState);
}


void XMem_MemoryRethink(MemoryData *pData)
{
	XYMem_MemoryRethink(&XMem, pData);
}


void	XMem_ROM_Write(Z80_WORD Port, Z80_BYTE Data)
{
	XYMem_ROM_Write(&XMem, Port, Data);
}

void	XMem_RAM_Write(Z80_WORD Port, Z80_BYTE Data)
{
	XYMem_RAM_Write(&XMem, Port, Data);
}

void	XMem_Flash_Write(Z80_WORD Port, Z80_BYTE Data)
{
	XYMem_Flash_Write(&XMem, Port, Data);
}


void XMem_Reset(void)
{
	XYMem_Reset(&XMem);
}


void XMem_InitDevice(void)
{
	XYMem_InitDevice(&XMem, TRUE);
}

CPCPortWrite XMemPortWrite[2] =
{
	{
		0x8100,            /* and */
		0x0100,            /* compare */
		XMem_RAM_Write
	},
	{
		0x02000,            /* and */
		0x00000,            /* compare */
		XMem_ROM_Write
	}
};

CPCPortWrite XMemMemWrite[1] =
{
	{
		0x0c000,            /* and */
		0x0c000,            /* compare */
		XMem_Flash_Write
	},
};




static EmuDeviceSwitch XMemSwitches[4] =
{
	{
		"Ram switch (On=6128, Off=464/664)",
		"RamSwitch",
		XMem_GetRamSwitch,
		XMem_SetRamSwitch
	},
	{
		"Boot (On=X-Mem, Off=CPC)",
		"Boot",
		XMem_GetBootSwitch,
		XMem_SetBootSwitch
	},
	{
		"Rom lock/protect (On=Yes, Off=No)",
		"RomProtect",
		XMem_GetRomProtectSwitch,
		XMem_SetRomProtectSwitch
	},
	{
		"Rom enabled (On=Yes, Off=No)",
		"RomEnable",
		XMem_GetRomEnabledSwitch,
		XMem_SetRomEnabledSwitch
	}
};


static EmuDevice XMemDevice=
{
	NULL,
	XMem_InitDevice,
	NULL,
	"XMEM",
	"XMem",
	"ToTo's X-Mem",
    CONNECTION_EXPANSION,   /* connects to expansion */
	DEVICE_FLAGS_HAS_DKTRONICS_RAM,                   
    0,                /* no read ports */
  NULL,
  sizeof(XMemPortWrite)/sizeof(XMemPortWrite[0]),                    /* 1 write ports */
  XMemPortWrite,
  0,                /* no memory read*/
  NULL,
  sizeof(XMemMemWrite)/sizeof(XMemMemWrite[0]),                /* no memory write */
  XMemMemWrite,
  XMem_Reset, /* no reset function */
  XMem_MemoryRethink, /* no rethink function */
  XMem_Reset, /* no power function */
	sizeof(XMemSwitches)/sizeof(XMemSwitches[0]),                      /* no switches */
	XMemSwitches,
    0,                      /* no buttons */
    NULL,
    0,                      /* no onboard roms */
    NULL,
	0,
	NULL,
	NULL,	/* no cursor update function */
	&XMem.m_Roms, /* rom slots */
	NULL,	/* printer */
	NULL, /* joystick */
	  0,
	  NULL,
	  NULL, /* sound */
	  NULL, /* lpen */
	  NULL, /* reti */
	  NULL, /* ack maskable interrupt */
	  &XMem.m_RamData, /* dkram data */
	  NULL, /* device ram */
	  NULL, /* device backup */
	  NULL
};

int XMem_Init(void)
{
	return RegisterDevice(&XMemDevice);
}

void YMem_MemoryRethink(MemoryData *pData)
{
	XYMem_MemoryRethink(&YMem, pData);
}

void YMem_Reset(void)
{
	XYMem_Reset(&YMem);
}
void YMem_InitDevice(void)
{
	XYMem_InitDevice(&YMem, FALSE);
}

void	YMem_ROM_Write(Z80_WORD Port, Z80_BYTE Data)
{
	XYMem_ROM_Write(&YMem, Port, Data);
}

void	YMem_RAM_Write(Z80_WORD Port, Z80_BYTE Data)
{
	XYMem_RAM_Write(&YMem, Port, Data);
}



BOOL YMem_GetRamSwitch()
{
	return XYMem_GetRamSwitch(&YMem);
}

void YMem_SetRamSwitch(BOOL bState)
{
	XYMem_SetRamSwitch(&YMem, bState);
}


BOOL YMem_GetBootSwitch(void)
{
	return XYMem_GetBootSwitch(&YMem);
}

void YMem_SetBootSwitch(BOOL bState)
{
	XYMem_SetBootSwitch(&YMem, bState);
}


BOOL YMem_GetRomProtectSwitch(void)
{
	return XYMem_GetRomProtectSwitch(&YMem);
}

void YMem_SetRomProtectSwitch(BOOL bState)
{
	XYMem_SetRomProtectSwitch(&YMem, bState);
}



BOOL YMem_GetRomEnabledSwitch(void)
{
	return XYMem_GetRomEnabledSwitch(&YMem);
}

void YMem_SetRomEnabledSwitch(BOOL bState)
{
	XYMem_SetRomEnabledSwitch(&YMem, bState);
}


CPCPortWrite YMemPortWrite[2] =
{
	{
		0x8100,            /* and */
		0x0000,            /* compare */
		YMem_RAM_Write
	},
	{
		0x02000,            /* and */
		0x00000,            /* compare */
		YMem_ROM_Write
	}
};



static EmuDeviceSwitch YMemSwitches[2] =
{
#if 0
	{
		"Ram switch (On=6128, Off=464/664)",
		"RamSwitch",
		YMem_GetRamSwitch,
		YMem_SetRamSwitch
	},
	{
		"Boot (On=X-Mem, Off=CPC)",
		"Boot",
		YMem_GetBootSwitch,
		YMem_SetBootSwitch
	},
#endif
	{
		"Rom lock/protect (On=Yes, Off=No)",
		"RomProtect",
		YMem_GetRomProtectSwitch,
		YMem_SetRomProtectSwitch
	},
	{
		"Rom enabled (On=Yes, Off=No)",
		"RomEnable",
		YMem_GetRomEnabledSwitch,
		YMem_SetRomEnabledSwitch
	}
};


static EmuDevice YMemDevice =
{
	NULL,
	YMem_InitDevice,
	NULL,
	"YMEM",
	"YMem",
	"ToTo's Y-Mem",
	CONNECTION_EXPANSION,   /* connects to expansion */
	DEVICE_FLAGS_HAS_DKTRONICS_RAM,
	0,                /* no read ports */
	NULL,
	sizeof(YMemPortWrite) / sizeof(YMemPortWrite[0]),                    /* 1 write ports */
	YMemPortWrite,
	0,                /* no memory read*/
	NULL,
	0,                /* no memory write */
	NULL,
	YMem_Reset, /* no reset function */
	YMem_MemoryRethink, /* no rethink function */
	YMem_Reset, /* no power function */
	sizeof(YMemSwitches) / sizeof(YMemSwitches[0]),                      /* no switches */
	YMemSwitches,
	0,                      /* no buttons */
	NULL,
	0, 
	NULL,
	0,                      /* no onboard roms */
	NULL,
	NULL,	/* no cursor update function */
	&YMem.m_Roms, /* rom slots */
	NULL,	/* printer */
	NULL, /* joystick */
	0,
	NULL,
	NULL, /* sound */
	NULL, /* lpen */
	NULL, /* reti */
	NULL, /* ack maskable interrupt */
	&YMem.m_RamData, /* dkram data */
	NULL, /* device ram */
	NULL, /* device backup */
	NULL
};

void YMem_Init(void)
{
	RegisterDevice(&YMemDevice);
}










